#!/bin/bash
set -e

# ====================================================================================
# Debian 13 Post-Install Setup Script
#
# This script performs the following actions:
# 1. Updates and upgrades the system.
# 2. Ensures 'sudo' is installed and the current user is in the sudo group.
# 3. Enables 'contrib', 'non-free', and 'non-free-firmware' repositories.
# 4. Installs wifi firmware (specifically for Intel iwlwifi).
# 5. Installs multimedia codecs and DVD playback libraries.
# 6. Sets up a weekly system cleanup cron job.
# 7. Configures CapsLock as an additional Ctrl key (GNOME).
# 8. Fixes SSH permissions for the user.
# ====================================================================================

# --- Step 0: Pre-flight checks ---

# Check if running as root
if [ "$EUID" -ne 0 ]; then
  echo "Error: This script must be run with sudo privileges."
  echo "Usage: sudo ./setup_debian.sh"
  exit 1
fi

if [ -z "$SUDO_USER" ]; then
    echo "Error: Could not detect the original user."
    echo "Please run this script using 'sudo' from your normal user account, not as root directly."
    exit 1
fi
REAL_USER="$SUDO_USER"
echo "Running setup for user: $REAL_USER"

# --- Step 1: System Update ---
echo ">>> Step 1: Updating system packages..."
# 'apt-get update' refreshes the package list
# 'apt-get upgrade -y' installs available upgrades for currently installed packages
apt-get update && apt-get upgrade -y

# --- Step 2: Enable sudo (if not already working) ---
echo ">>> Step 2: Ensuring sudo is installed and configured..."
# Install sudo package
apt-get install -y sudo

# Add the user to the 'sudo' group if not already a member
# 'usermod -aG' appends the group to the user's list of groups
if ! groups "$REAL_USER" | grep -q "\bsudo\b"; then
    echo "Adding $REAL_USER to sudo group..."
    usermod -aG sudo "$REAL_USER"
    echo "NOTE: You may need to logout and login again for sudo group membership to take effect."
fi

# --- Step 3: Enable non-free repositories ---
echo ">>> Step 3: Enabling contrib, non-free, and non-free-firmware repositories..."
# Backup the current sources list
cp /etc/apt/sources.list /etc/apt/sources.list.bak

# Update sources.list to include contrib, non-free, and non-free-firmware
# This ensures access to proprietary drivers and firmware often needed for hardware support
# We use 'sed' to append these components to lines starting with 'deb' that explicitly mention 'main'
# but don't yet have 'non-free-firmware'.
# A simple approach is adding them to all 'main' lines if not present.
sed -i 's/main/main contrib non-free non-free-firmware/' /etc/apt/sources.list

# Update package lists with the new repositories
apt-get update

# --- Step 4: Wifi Firmware ---
echo ">>> Step 4: Installing wifi firmware..."
# Installing 'firmware-iwlwifi' for Intel wireless cards.
# If you have a different card, you might need a different package (e.g., firmware-realtek).
apt-get install -y firmware-iwlwifi nextcloud-desktop

# --- Step 5: Multimedia Support ---
echo ">>> Step 5: Installing multimedia codecs and DVD playback tools..."
# 'libdvd-pkg' automates downloading and installing libdvdcss for DVD playback
# 'regionset' allows setting the DVD region code
# 'libavcodec-extra' provides additional codecs for various media formats
# 'vlc' is a versatile media player
apt-get install -y libdvd-pkg regionset libavcodec-extra vlc

# Reconfigure libdvd-pkg to build and install the necessary libraries
# DEBIAN_FRONTEND=noninteractive prevents blocking prompts
echo "Configuring libdvd-pkg..."
DEBIAN_FRONTEND=noninteractive dpkg-reconfigure libdvd-pkg

# --- Step 6: Cleanup Script & Cron Job ---
echo ">>> Step 6: Setting up weekly cleanup script..."

CLEANUP_SCRIPT="/usr/local/bin/local-cleanup"

# Create the cleanup script file
cat <<EOF > "$CLEANUP_SCRIPT"
#!/bin/bash
# Maintenance script to clean logs and package cache
# Adapted for automated execution as root

# Function to get current used space on / in blocks
get_used_space() {
    df / | awk 'NR==2 {print \$3}'
}

# Capture space BEFORE cleanup
PRE_CLEAN=\$(get_used_space)

echo "Starting system cleanup..."

# Reduces journal logs to 300 MB
journalctl --vacuum-size=300M

# Forces rotation of system logs
logrotate /etc/logrotate.conf

# Deletes downloaded package files (.deb)
apt-get clean

# Removes unused packages
apt-get autoremove -y

# Capture space AFTER cleanup
POST_CLEAN=\$(get_used_space)

# Calculate total freed space (in KB, then convert to MB/GB)
FREED_KB=\$((PRE_CLEAN - POST_CLEAN))
FREED_HUMAN=\$(echo "\$FREED_KB" | numfmt --from-unit=1024 --to=iec)

echo "Total disk space freed: \$FREED_HUMAN"

# Determine the primary user (assumed UID 1000) for home directory stats
TARGET_USER=\$(id -un 1000)
TARGET_HOME=\$(eval echo ~\$TARGET_USER)

echo "--- 10 Largest Files in \$TARGET_USER's Home (\$TARGET_HOME) ---"
if [ -d "\$TARGET_HOME" ]; then
    find "\$TARGET_HOME" -type f -printf "%s %p\n" 2>/dev/null | sort -rn | head -n 10 | numfmt --to=iec --field=1
else
    echo "Home directory not found: \$TARGET_HOME"
fi

echo "--- 10 Largest Files on Root Filesystem (Excluding /home & other mounts) ---"
find / -xdev -type f -printf "%s %p\n" 2>/dev/null | sort -rn | head -n 10 | numfmt --to=iec --field=1

echo "Maintenance complete."
EOF

# Make the script executable
chmod +x "$CLEANUP_SCRIPT"

# Link it to cron.weekly for automatic execution
ln -sf "$CLEANUP_SCRIPT" /etc/cron.weekly/local-cleanup
echo "Cleanup script installed to $CLEANUP_SCRIPT and scheduled weekly."



# --- Step 7: SSH Permissions (User Level) ---
echo ">>> Step 7: Fixing SSH permissions..."
# Ensure .ssh directory has correct permissions (700)
# Ensure private keys (id_rsa) have 600, public keys (id_rsa.pub) have 644
su - "$REAL_USER" -c "
mkdir -p ~/.ssh
chmod 700 ~/.ssh
touch ~/.ssh/authorized_keys
chmod 600 ~/.ssh/authorized_keys
if [ -f ~/.ssh/id_rsa ]; then chmod 600 ~/.ssh/id_rsa; fi
if [ -f ~/.ssh/id_rsa.pub ]; then chmod 644 ~/.ssh/id_rsa.pub; fi
"


# --- Step 8: Keyboard Settings (User Level) ---
echo ">>> Step 8: Configuring keyboard settings (Interactive)..."

echo ""
echo "=========================================================="
echo "The following keyboard changes will be made:"
echo "- Layouts: US and German (Switch with Win+Space)"
echo "- CapsLock: Remapped to additional Ctrl"
echo "- Shortcuts: Ctrl+PgUp opens Terminal"
echo "=========================================================="
echo ""

read -p "Do you want to apply these keyboard settings? (y/N) " response
if [[ "\$response" =~ ^[Yy]$ ]]; then
    echo "Applying keyboard settings..."
    # We need to run gsettings as the user.
    su - "$REAL_USER" -c "
        # 1. Set Keyboard Layouts (US and German)
        gsettings set org.gnome.desktop.input-sources sources \"[('xkb', 'us'), ('xkb', 'de')]\"

        # 2. Make CapsLock an additional Ctrl key
        gsettings set org.gnome.desktop.input-sources xkb-options \"['ctrl:nocaps']\"

        # 3. Add Custom Shortcut for Terminal (Ctrl+PgUp)
        KEY_PATH=\"/org/gnome/settings-daemon/plugins/media-keys/custom-keybindings/custom-terminal/\"

        # Set properties
        gsettings set org.gnome.settings-daemon.plugins.media-keys.custom-keybinding:\$KEY_PATH name 'GNOME Terminal'
        gsettings set org.gnome.settings-daemon.plugins.media-keys.custom-keybinding:\$KEY_PATH command 'gnome-terminal'
        gsettings set org.gnome.settings-daemon.plugins.media-keys.custom-keybinding:\$KEY_PATH binding '<Control>Page_Up'

        # Append to list of custom keybindings
        CURRENT=\$(gsettings get org.gnome.settings-daemon.plugins.media-keys custom-keybindings)
        if [ \"\$CURRENT\" = \"@as []\" ] || [ \"\$CURRENT\" = \"[]\" ]; then
            gsettings set org.gnome.settings-daemon.plugins.media-keys custom-keybindings \"['\$KEY_PATH']\"
        elif ! echo \"\$CURRENT\" | grep -q \"custom-terminal\"; then
            # Append logic: remove closing bracket, add comma and new path
            NEW=\${CURRENT%]}
            gsettings set org.gnome.settings-daemon.plugins.media-keys custom-keybindings \"\${NEW}, '\$KEY_PATH']\"
        fi
    " || echo "Warning: Keyboard setup failed (requires GNOME session or dconf)."
else
    echo "Skipping keyboard configuration."
fi

echo "=========================================================="
echo "Setup Complete! Please reboot your machine to finalize changes."
echo "=========================================================="
